/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "swap.h"


gboolean
swap_new (SwapFile * swap,
          gchar *    filename)
{
  g_return_val_if_fail (swap != NULL, FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);

  swap->filename = g_strdup (filename);
  swap->size = 0;
  swap->fd = -1;
  swap->offset = -1;

  return TRUE;
}


gboolean
swap_delete (SwapFile * swap)
{
  return TRUE;
}


gboolean
swap_open (SwapFile * swap)
{
  g_return_val_if_fail (swap != NULL, FALSE);

  if (swap->fd == -1)
    {
      swap->fd = open (swap->filename,
                       O_CREAT|O_RDWR,
                       S_IRUSR|S_IWUSR);
      g_return_val_if_fail (swap->fd != -1, FALSE);
      swap->offset = 0;
    }
  
  return TRUE;
}


gboolean
swap_close (SwapFile * swap)
{
  g_return_val_if_fail (swap != NULL, FALSE);

  if (swap->fd != -1)
    {
      g_return_val_if_fail (close (swap->fd) != -1, FALSE);
      swap->fd = -1;
      swap->offset = -1;
   }
  
  return TRUE;
}


SwapTag
swap_alloc (SwapFile * swap,
            gint       size)
{
  g_return_val_if_fail (swap != NULL, -1);
  {
    gint rc = swap->size;
    swap->size += size;
    return rc;
  }
}


gboolean
swap_free (SwapFile * swap,
           SwapTag    tag)
{
  return TRUE;  
}


gboolean
swap_read (SwapFile * swap,
           SwapTag    tag,
           gchar *    buffer,
           gint       size)
{
  gint err;
      
  g_return_val_if_fail (swap != NULL, FALSE);
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (tag >= 0, FALSE);
  g_return_val_if_fail (size > 0, FALSE);
  g_return_val_if_fail (tag+size <= swap->size, FALSE);

  if (swap->fd == -1)
    {
      g_return_val_if_fail (swap_open (swap) == TRUE, FALSE);
    }
  
  if (swap->offset != tag)
    {
      tag = lseek (swap->fd, tag, SEEK_SET);
      g_return_val_if_fail (tag != -1, FALSE);
      swap->offset = tag;
    }
  
  while (size)
    {
      do {
	err = read (swap->fd, buffer, size);
      } while ((err == -1) && ((errno == EAGAIN) || (errno == EINTR)));
      
      g_return_val_if_fail (err > 0, FALSE);      
      swap->offset += err;
      size -= err;
      buffer += err;
    }

  return TRUE;
}


gboolean
swap_write (SwapFile * swap,
            SwapTag    tag,
            gchar *    buffer,
            gint       size)
{
  gint err;
      
  g_return_val_if_fail (swap != NULL, FALSE);
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (tag >= 0, FALSE);
  g_return_val_if_fail (size > 0, FALSE);
  g_return_val_if_fail (tag+size <= swap->size, FALSE);

  if (swap->fd == -1)
    {
      g_return_val_if_fail (swap_open (swap) == TRUE, FALSE);
    }

  if (swap->offset != tag)
    {
      tag = lseek (swap->fd, tag, SEEK_SET);
      g_return_val_if_fail (tag != -1, FALSE);
      swap->offset = tag;
    }

  while (size)
    {
      do {
	err = write (swap->fd, buffer, size);
      } while ((err == -1) && ((errno == EAGAIN) || (errno == EINTR)));
      
      g_return_val_if_fail (err > 0, FALSE);
      swap->offset += err;
      size -= err;
      buffer += err;
    }

  return TRUE;
}


