/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef __MEMORY_H__
#define __MEMORY_H__

#include <glib.h>
#include "swap.h"

/*
    These macros are the user interface to the COW memory manager.
    Instead of using a raw pointer, you create a GimpMemPtr object
    (which is just a smart pointer) and operate through that.  The
    smart pointer has a copy of the real data pointer.  This copy is
    NULL if the readwrite/release balance is zero.

    The smart pointer never moves (once you have a pointer to it, the
    pointer is always valid).  However, the copy of the real data
    pointer (inside the smart pointer) can change after ANY operation
    using the smart pointer.  Always use the macros to read it so you
    get the newest value.

    The first set (d_*) are commands that operate on the smart
    pointer.  They must appear on the RHS.

    The second set (*_d) are different casts of the real data pointer.
    These are okay on either the LHS or RHS.

    q    - GimpMemPtr *
    new  - GimpMemPtr *
    size - int
*/

/* RHS expressions */
#define   d_init(q)              (gimp_memptr_init((q)))
#define   d_uninit(q)            (gimp_memptr_uninit((q)))
#define   d_alloc(q,size)        (gimp_memptr_alloc((q),(size)))
#define   d_unalloc(q)           (gimp_memptr_unalloc((q)))
#define   d_read(q)              (gimp_memptr_use((q),FALSE,FALSE))
#define   d_write(q)             (gimp_memptr_use((q),TRUE,TRUE))
#define   d_update(q)            (gimp_memptr_use((q),TRUE,FALSE))
#define   d_release(q)           (gimp_memptr_unuse((q)))
#define   d_join(q,new)          (gimp_memptr_join((q),(new)))
#define   d_split(q)             (gimp_memptr_split((q)))
#define   d_is_alloced(q)        ((q)->mem != NULL)
#define   d_is_joined(q)         ((q)->joined == TRUE)
#define   d_usecount(q)          ((q)->usecount)
#define   d_size(q)              ((q)->size)

/* LHS expressions */
#define   void_d(q)             ((q)->data)
#define   gchar_d(q)            ((gchar*)(q)->data)
#define   guchar_d(q)           ((guchar*)(q)->data)
#define   guint8_d(q)           ((guint8*)(q)->data)
#define   guint16_d(q)          ((guint16*)(q)->data)
#define   gfloat_d(q)           ((gfloat*)(q)->data)



typedef struct _GimpMem       GimpMem;
typedef struct _GimpMemCache  GimpMemCache;
typedef struct _GimpMemList   GimpMemList;
typedef struct _GimpMemPtr    GimpMemPtr;
typedef struct _GimpMemSwap   GimpMemSwap;


/* a list of GimpMem objects and their total size */
struct _GimpMemList
{
  GimpMem * head;
  GimpMem * tail;
  gint len;
  gint size;
};


/* cache info for a GimpMem */
struct _GimpMemCache
{
  GimpMemList * list;
  GimpMem * next;
  GimpMem * prev;
};


/* swap info for a GimpMem */
struct _GimpMemSwap
{
  SwapFile * file;
  SwapTag    tag;
  gboolean   dirty;
};


/* a piece of sharable swappable image memory.  data may change when
   usecount touches 0 */
struct _GimpMem
{
  /* the actual data */
  void * data;

  /* length in bytes */
  gint size;

  /* the number of ptrs attached to us */
  gint ptrcount;

  /* the number of ptrs with at least one outstanding use */
  gint usecount;

  /* will we allow another ptr to attach? */
  gboolean sharable;

  /* cache info */
  GimpMemCache cache;

  /* swap info */
  GimpMemSwap swap;
};


/* a pointer to a piece of sharable swappable memory.  'mem' and
   'data' are volatile, and may change after any operation on this
   object.  'data' will be NULL if usecount == 0 */
struct _GimpMemPtr
{
  /* the GimpMem we are associated with */
  GimpMem * mem;

  /* the number of local refs */
  gint usecount;

  /* might we be sharing our GimpMem? */
  gboolean joined;

  /* a local copy of the data size */
  gint size;
  
  /* a local copy of the data pointer in the GimpMem */
  void * data;
};


void      gimp_memory_init      (gchar       *filename);
void      gimp_memptr_init      (GimpMemPtr  *ptr);
void      gimp_memptr_uninit    (GimpMemPtr  *ptr);
gboolean  gimp_memptr_alloc     (GimpMemPtr  *ptr,
                                 gint         size);
gboolean  gimp_memptr_unalloc   (GimpMemPtr  *ptr);
gboolean  gimp_memptr_use       (GimpMemPtr  *ptr,
                                 gboolean     dirty,
                                 gboolean     cow);
gboolean  gimp_memptr_unuse     (GimpMemPtr  *ptr);
gboolean  gimp_memptr_join      (GimpMemPtr  *ptr,
                                 GimpMemPtr  *newptr);
gboolean  gimp_memptr_split     (GimpMemPtr  *ptr);


#endif /* __MEMORY_H__ */
